/*
 * Decompiled with CFR 0.152.
 */
package de.xam.files;

import com.google.common.base.Function;
import com.google.common.collect.Sets;
import com.ibm.icu.text.CharsetDetector;
import com.ibm.icu.text.CharsetMatch;
import de.xam.files.FilenameUtils;
import de.xam.files.ReadersAndWriters;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import javax.swing.filechooser.FileSystemView;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

public class FileTools {
    public static final long A_WEEK = 604800000L;
    public static final FileFilter ALL_FILES_AND_DIRS_FILTER = new FileFilter(){

        @Override
        public boolean accept(File pathname) {
            return true;
        }
    };
    public static final FileFilter ALL_FILES_NO_DIRS_FILTER = new FileFilter(){

        @Override
        public boolean accept(File pathname) {
            return pathname.isFile();
        }
    };
    private static CharsetDetector charsetDetector;
    private static final Set<String> EXTENSIONS_TEXT;
    public static final FileFilter FILEFILTER_ALL_DIRECTORIES;
    public static final String[] GENERATED_FILES_ENDINGS;
    public static final String[] GENERATED_FILES_PARTS;
    public static final FileFilter IGNORE_GENERATED_FILES;
    public static final FileFilter IGNORE_GENERATED_FILES_AND_DIRS;
    private static Logger log;
    public static final FileFilter NO_FILES_ALL_DIRS_FILTER;

    public static FileFilter and(final FileFilter ... filters) {
        assert (filters != null && filters.length >= 1);
        return new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                for (FileFilter ff : filters) {
                    if (ff.accept(pathname)) continue;
                    return false;
                }
                return true;
            }
        };
    }

    public static String changeExtension(String path, String sourceExt, String targetExt) throws IllegalArgumentException {
        if (!path.endsWith(sourceExt)) {
            throw new IllegalArgumentException("Path does not end with old extension");
        }
        return path.substring(0, path.length() - sourceExt.length()) + targetExt;
    }

    public static String changeExtensionIfPossible(String path, String sourceExt, String targetExt) {
        if (path.endsWith(sourceExt)) {
            return path.substring(0, path.length() - sourceExt.length()) + targetExt;
        }
        return path;
    }

    public static long changesBefore(File file, long before) {
        return Math.min(before, file.lastModified());
    }

    public static boolean contains(File rootPath, File subPath) {
        return FileTools.contains(rootPath.getAbsolutePath(), subPath.getAbsolutePath());
    }

    public static boolean contains(File rootPath, String subPath) {
        return FileTools.contains(rootPath.getAbsolutePath(), subPath);
    }

    public static boolean contains(String absoluteCanonicalPathA, String absoluteCanonicalPathB) {
        return absoluteCanonicalPathB.startsWith(absoluteCanonicalPathA);
    }

    public static void copyDirAndPreservePermissions(final File from, final File to) throws IOException {
        FileTools.process(from, ALL_FILES_AND_DIRS_FILTER, new IFileProcessor(){

            @Override
            public void process(File f) throws IOException {
                String sourceRelativePath = FileTools.getRootRelativePath(from, f);
                File target = new File(to, sourceRelativePath);
                target.getParentFile().mkdirs();
                Files.copy(f.toPath(), target.toPath(), StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
            }
        }, null, true);
    }

    public static void copyDirInSyncMode(final File srcDir, final File targetDir) throws IOException {
        FileTools.process(srcDir, IGNORE_GENERATED_FILES_AND_DIRS, new IFileProcessor(){

            @Override
            public void process(File srcFile) throws IOException {
                String rel = FileTools.getRootRelativePath(srcDir, srcFile);
                File targetFile = new File(targetDir, rel);
                targetFile.getParentFile().mkdirs();
                if (!targetFile.exists() || targetFile.lastModified() < srcFile.lastModified()) {
                    log.info("Copying " + srcFile.getCanonicalPath());
                    FileUtils.copyFile((File)srcFile, (File)targetFile);
                }
            }
        }, null, true);
    }

    public static void copyTranscoded(File inFile, String inputEncoding, File outFile, String outputEncoding) throws IOException {
        FileInputStream is = new FileInputStream(inFile);
        InputStreamReader input = new InputStreamReader((InputStream)is, inputEncoding);
        FileOutputStream os = new FileOutputStream(outFile);
        OutputStreamWriter output = new OutputStreamWriter((OutputStream)os, outputEncoding);
        IOUtils.copy((Reader)input, (Writer)output);
        output.flush();
        output.close();
        input.close();
    }

    public static void deleteDirCarefully(File dir, long maxAgeInMillis) throws IOException, IllegalStateException {
        if (!dir.exists()) {
            return;
        }
        long now = System.currentTimeMillis();
        long aWhileAgo = now - maxAgeInMillis;
        long oldest = FileTools.dirChangesBefore(dir, new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                return true;
            }
        }, aWhileAgo);
        if (oldest < aWhileAgo) {
            long delta = now - oldest;
            long age = delta / 86400000L;
            throw new IllegalStateException("Refusing to delete " + dir.getAbsolutePath() + " there are files older than " + maxAgeInMillis + " (ca. " + age + " days)!");
        }
        log.info("Deleting " + dir.getAbsolutePath());
        FileUtils.deleteDirectory((File)dir);
    }

    private static void deleteFile(File f) {
        f.delete();
        for (int timeout = 50; f.exists() && timeout < 1000; timeout *= 2) {
            f.delete();
            try {
                Thread.sleep(timeout);
                continue;
            }
            catch (InterruptedException e) {
                log.warn("Could not sleep");
            }
        }
        if (f.exists()) {
            throw new IllegalStateException("Could not delete " + f.getAbsolutePath());
        }
    }

    public static long dirChangesBefore(File dir, FileFilter fileFilter, long before) {
        assert (dir != null);
        assert (dir.exists());
        long min = FileTools.changesBefore(dir, before);
        File[] files = dir.listFiles(fileFilter);
        if (files != null) {
            for (File f : files) {
                min = FileTools.changesBefore(f, min);
            }
        }
        if ((files = dir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File f) {
                return f.isDirectory();
            }
        })) != null) {
            for (File f : files) {
                min = FileTools.dirChangesBefore(f, fileFilter, min);
            }
        }
        return min;
    }

    public static File disambiguate(File desiredFile) {
        int fileNo = 2;
        File usedFile = desiredFile;
        while (usedFile.exists()) {
            usedFile = new File(desiredFile.getAbsolutePath() + "-" + fileNo++);
        }
        return usedFile;
    }

    public static List<File> getDirectoryEntries(File dir) {
        assert (dir != null);
        assert (dir.isDirectory());
        log.debug("Reading " + dir.getAbsolutePath());
        if (!dir.isDirectory()) {
            throw new RuntimeException(dir.getAbsolutePath() + " is not a directory.");
        }
        ArrayList<File> directoryEntries = new ArrayList<File>();
        File[] files = dir.listFiles();
        if (files == null) {
            log.warn("Could not read dir entries of '" + dir.getAbsolutePath() + "'");
        } else {
            directoryEntries.addAll(Arrays.asList(files));
        }
        return directoryEntries;
    }

    private static File[] getDirectoryEntriesAsArray(File dir) {
        assert (dir != null);
        assert (dir.isDirectory());
        log.trace("Reading " + dir.getAbsolutePath());
        if (!dir.isDirectory()) {
            throw new RuntimeException(dir.getAbsolutePath() + " is not a directory.");
        }
        File[] files = dir.listFiles();
        return files;
    }

    public static File[] getDirectoryEntriesAsSortedArray(File dir) {
        File[] files = FileTools.getDirectoryEntriesAsArray(dir);
        if (files == null) {
            log.warn("Could not read dir entries of '" + dir.getAbsolutePath() + "'");
            return files;
        }
        Arrays.sort(files, new Comparator<File>(){

            @Override
            public int compare(File o1, File o2) {
                return o1.getAbsolutePath().compareTo(o2.getAbsolutePath());
            }
        });
        return files;
    }

    public static List<File> getDirectoryEntriesAsSortedList(File dir) {
        List<File> directoryEntries = FileTools.getDirectoryEntries(dir);
        Collections.sort(directoryEntries, new Comparator<File>(){

            @Override
            public int compare(File o1, File o2) {
                return o1.getAbsolutePath().compareTo(o2.getAbsolutePath());
            }
        });
        return directoryEntries;
    }

    public static String getDisplayName(File f) {
        FileSystemView v = FileSystemView.getFileSystemView();
        return v.getSystemDisplayName(f);
    }

    public static String getNameWithoutExtension(File f) {
        String s = f.getName();
        int i = s.lastIndexOf(".");
        if (i == -1) {
            return s;
        }
        return s.substring(0, i);
    }

    public static String getOsDisplayFileName(File f) {
        FileSystemView v = FileSystemView.getFileSystemView();
        return v.getSystemDisplayName(f);
    }

    public static String getOsDisplayVolumeLabel(File anyFileOnThatVolume) {
        return FileTools.getOsDisplayFileName(FileTools.getRootFolder(anyFileOnThatVolume));
    }

    public static File getRootFolder(File f) {
        if (!f.exists()) {
            throw new IllegalArgumentException(f.getAbsolutePath() + " does not exist");
        }
        File parent = f.getParentFile();
        if (parent == null) {
            return f;
        }
        return FileTools.getRootFolder(parent);
    }

    public static String getRootRelativePath(File root, File f) {
        assert (root != null);
        assert (!root.getAbsolutePath().endsWith("/"));
        assert (f.getAbsolutePath().startsWith(root.getAbsolutePath()));
        return f.getAbsolutePath().substring(root.getAbsolutePath().length());
    }

    public static String getVolumeLabel(File anyFileOnThatVolume) {
        return FileTools.getDisplayName(FileTools.getRootFolder(anyFileOnThatVolume));
    }

    public static synchronized String guessEncoding(File f) throws IOException {
        assert (f.exists());
        assert (f.isFile());
        if (charsetDetector == null) {
            charsetDetector = new CharsetDetector();
        }
        CharsetMatch match = null;
        try (FileInputStream in = null;){
            in = new FileInputStream(f);
            BufferedInputStream bis = new BufferedInputStream(in, 1024);
            charsetDetector.setText((InputStream)bis);
            match = charsetDetector.detect();
            bis.close();
        }
        if (match == null) {
            return null;
        }
        int confidence = match.getConfidence();
        if (confidence < 60) {
            return null;
        }
        return match.getName();
    }

    public static FileAndAge lastModifiedMax(File fileOrDir, FileFilter fileFilter) {
        try {
            BasicFileAttributes atts = Files.readAttributes(fileOrDir.toPath(), BasicFileAttributes.class, new LinkOption[0]);
            if (atts.isRegularFile()) {
                return new FileAndAge(fileOrDir, atts.lastModifiedTime().toMillis());
            }
            FileAndAge max = new FileAndAge(null, 0L);
            File[] subFileOrDirs = fileOrDir.listFiles(fileFilter);
            if (subFileOrDirs != null) {
                for (File f : subFileOrDirs) {
                    FileAndAge sub = FileTools.lastModifiedMax(f, fileFilter);
                    if (sub.age <= max.age) continue;
                    max = sub;
                }
            }
            return max;
        }
        catch (IOException e) {
            log.warn("Could not read file attributes of '" + fileOrDir.getAbsolutePath() + "'");
            return new FileAndAge(null, 0L);
        }
    }

    public static FileAndAge lastModifiedMin(File fileOrDir, FileFilter fileFilter) {
        if (fileOrDir.isFile()) {
            return new FileAndAge(fileOrDir, fileOrDir.lastModified());
        }
        FileAndAge min = new FileAndAge(null, Long.MAX_VALUE);
        for (File f : fileOrDir.listFiles(fileFilter)) {
            FileAndAge sub = FileTools.lastModifiedMin(f, fileFilter);
            if (sub.age >= min.age) continue;
            min = sub;
        }
        return min;
    }

    public static boolean looksLikeATextFile(File f) {
        return EXTENSIONS_TEXT.contains(FilenameUtils.getExtension(f));
    }

    public static void main(String[] args) throws IOException {
        File dir = new File("/Users/xamde/_data_/_p_/_git/DwzGit/de.xam.files/src/main/java/de/xam/files");
        File zipTargetFile = new File("/Users/xamde/_data_/_p_/_git/DwzGit/de.xam.files/target/zips/test.zip");
        File unzipTargetDir = new File("/Users/xamde/_data_/_p_/_git/DwzGit/de.xam.files/target/unzips");
        FileTools.zip(dir, ALL_FILES_AND_DIRS_FILTER, zipTargetFile, null);
        System.out.println("------");
        FileTools.unzip(zipTargetFile, unzipTargetDir);
    }

    public static List<File> mostRecentKFiles(File dir, final int k, FileFilter fileFilter) throws IOException {
        final TreeMap mostRecent = new TreeMap();
        FileTools.process(dir, fileFilter, new IFileProcessor(){

            @Override
            public void process(File f) throws IOException {
                long modified = f.lastModified();
                if (mostRecent.size() < k) {
                    mostRecent.put(modified, f);
                } else if ((Long)mostRecent.firstKey() < modified) {
                    mostRecent.put(modified, f);
                    mostRecent.remove(mostRecent.firstKey());
                }
            }
        }, null, false);
        ArrayList<File> files = new ArrayList<File>();
        files.addAll(mostRecent.values());
        Collections.reverse(files);
        return files;
    }

    public static void process(File startDirectory, FileFilter fileFilter, IFileProcessor fileProcessor, IDirProcessor dirProcessor, boolean processSymbolicLinks) throws IOException {
        File[] filesAndDirs;
        assert (startDirectory != null);
        assert (fileFilter != null);
        assert (startDirectory.isDirectory());
        if (dirProcessor != null) {
            dirProcessor.process(startDirectory);
        }
        if (fileProcessor != null) {
            for (File f : startDirectory.listFiles(fileFilter)) {
                if (!f.isFile()) continue;
                try {
                    fileProcessor.process(f);
                }
                catch (Error | RuntimeException e) {
                    throw new RuntimeException("While processing " + f.getAbsolutePath(), e);
                }
            }
        }
        if ((filesAndDirs = startDirectory.listFiles(FILEFILTER_ALL_DIRECTORIES)) != null) {
            for (File f : filesAndDirs) {
                if (!processSymbolicLinks && Files.isSymbolicLink(f.toPath())) continue;
                FileTools.process(f, fileFilter, fileProcessor, dirProcessor, processSymbolicLinks);
            }
        }
    }

    public static void process(File[] startDirectories, FileFilter fileFilter, IFileProcessor fileProcessor) throws IOException {
        FileTools.process(startDirectories, fileFilter, fileProcessor, null, true);
    }

    public static void process(File[] startDirectories, FileFilter fileFilter, IFileProcessor fileProcessor, IDirProcessor dirProcessor, boolean processSymbolicLinks) throws IOException {
        assert (startDirectories != null);
        assert (fileFilter != null);
        for (File startDirectory : startDirectories) {
            FileTools.process(startDirectory, fileFilter, fileProcessor, dirProcessor, processSymbolicLinks);
        }
    }

    public static String readFileToStringWithBestEncoding(File f) throws IOException {
        assert (f.exists()) : "not found " + f.getAbsolutePath();
        assert (f.isFile());
        String encoding = FileTools.guessEncoding(f);
        if (encoding == null) {
            encoding = "utf-8";
        }
        return FileUtils.readFileToString((File)f, (String)encoding);
    }

    public static void renameToAndOverwriteIfNecessary(File source, File target) {
        if (target.exists()) {
            FileTools.deleteFile(target);
        }
        assert (!target.exists());
        source.renameTo(target);
    }

    public static void searchAndReplace(File f, String search, String replace) throws IOException {
        log.info("Replacing '" + search + "' with '" + replace + "' in " + f.getAbsolutePath());
        BufferedReader br = new BufferedReader(ReadersAndWriters.getUtf8Reader(f));
        File tempFile = File.createTempFile("FileTools", ".temp");
        Writer w = ReadersAndWriters.getUtf8Writer(tempFile);
        String line = br.readLine();
        while (line != null) {
            line = line.replace(search, replace);
            w.write(line + "\n");
            line = br.readLine();
        }
        w.close();
        br.close();
        FileTools.renameToAndOverwriteIfNecessary(tempFile, f);
    }

    public static void searchAndReplaceRecursive(File root, FileFilter fileFilter, final String search, final String replace, boolean followSymbolicLinks) throws IOException {
        FileTools.process(root, fileFilter, new IFileProcessor(){

            @Override
            public void process(File f) throws IOException {
                FileTools.searchAndReplace(f, search, replace);
            }
        }, null, followSymbolicLinks);
    }

    public static boolean targetIsUpToDate(File src, File target) {
        if (!target.exists()) {
            return false;
        }
        assert (src.exists());
        return FileTools.lastModifiedMax((File)src, (FileFilter)FileTools.IGNORE_GENERATED_FILES_AND_DIRS).age <= FileTools.lastModifiedMin((File)target, (FileFilter)FileTools.IGNORE_GENERATED_FILES_AND_DIRS).age;
    }

    public static File[] topDirs(File rootDir, File subDir) {
        assert (rootDir.isDirectory());
        assert (subDir.isDirectory()) : "" + subDir.getAbsolutePath();
        assert (FileTools.contains(rootDir, subDir));
        ArrayList<File> top = new ArrayList<File>();
        File current = subDir;
        while (!current.equals(rootDir)) {
            top.add(0, current);
            current = current.getParentFile();
        }
        top.add(0, current);
        return top.toArray(new File[top.size()]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void zip(File sourceDir, FileFilter includeFiter, File targetFile, Long fixedFiledate) throws IOException {
        assert (sourceDir.exists());
        assert (sourceDir.isDirectory());
        if (targetFile.exists()) {
            assert (targetFile.isFile());
            targetFile.delete();
        } else {
            targetFile.getParentFile().mkdirs();
        }
        final ArrayList list = new ArrayList();
        FileTools.process(sourceDir, includeFiter, new IFileProcessor(){

            @Override
            public void process(File f) throws IOException {
                list.add(f);
            }
        }, new IDirProcessor(){

            @Override
            public void process(File dir) throws IOException {
                list.add(dir);
            }
        }, false);
        byte[] buffer = new byte[65536];
        FileOutputStream fos = null;
        ZipOutputStream zos = null;
        try {
            fos = new FileOutputStream(targetFile);
            zos = new ZipOutputStream(fos);
            zos.setLevel(9);
            for (File fileOrDir : list) {
                String path = FileTools.getRootRelativePath(sourceDir, fileOrDir);
                if (fileOrDir.isDirectory()) {
                    path = path + "/";
                }
                ZipEntry ze = new ZipEntry(path);
                ze.setTime(fixedFiledate == null ? fileOrDir.lastModified() : fixedFiledate.longValue());
                zos.putNextEntry(ze);
                FileInputStream in = null;
                try {
                    if (!fileOrDir.isFile()) continue;
                    in = new FileInputStream(fileOrDir);
                    IOUtils.copyLarge((InputStream)in, (OutputStream)zos, (byte[])buffer);
                }
                finally {
                    if (in == null) continue;
                    in.close();
                }
            }
            zos.closeEntry();
            log.trace("Folder " + sourceDir.getAbsolutePath() + "successfully compressed");
        }
        catch (IOException ex) {
            ex.printStackTrace();
        }
        finally {
            try {
                if (zos != null) {
                    zos.close();
                }
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public static InputStream readFromZip(File zipFile, String entryName) throws IOException {
        final FileInputStream fis = new FileInputStream(zipFile);
        ZipInputStream zis = new ZipInputStream(fis);
        ZipEntry ze = zis.getNextEntry();
        while (ze != null) {
            if (ze.getName().equals("/" + entryName)) {
                WrappedInputStream wis = new WrappedInputStream(zis, new Function<Void, Void>(){

                    public Void apply(Void input) {
                        try {
                            fis.close();
                        }
                        catch (IOException e) {
                            throw new RuntimeException("Error", e);
                        }
                        return null;
                    }
                });
                return wis;
            }
            ze = zis.getNextEntry();
        }
        throw new RuntimeException("no entry named '" + entryName + "' found in zip file " + zipFile.getAbsolutePath());
    }

    public static void unzip(File zipFile, File targetDir) throws IOException {
        assert (zipFile.exists());
        targetDir.mkdirs();
        FileInputStream fis = new FileInputStream(zipFile);
        ZipInputStream zis = new ZipInputStream(fis);
        byte[] buffer = new byte[65536];
        ZipEntry ze = zis.getNextEntry();
        while (ze != null) {
            String relativePathName = ze.getName();
            long lastModified = ze.getTime();
            if (!relativePathName.equals("/")) {
                String fileName = targetDir.getAbsolutePath() + "/" + (relativePathName.startsWith("/") ? relativePathName.substring(1) : relativePathName);
                File fileOrDir = new File(fileName);
                if (relativePathName.endsWith("/")) {
                    fileOrDir.mkdirs();
                } else {
                    fileOrDir.getParentFile().mkdirs();
                    FileOutputStream fos = new FileOutputStream(fileOrDir);
                    IOUtils.copyLarge((InputStream)zis, (OutputStream)fos, (byte[])buffer);
                    fos.flush();
                    fos.close();
                }
                if (lastModified >= 0L) {
                    fileOrDir.setLastModified(lastModified);
                }
            }
            ze = zis.getNextEntry();
        }
        zis.close();
        fis.close();
    }

    public static Properties readPropertiesFromUtf8File(File f) throws IOException {
        assert (f != null);
        assert (f.exists());
        Properties props = new Properties();
        try (Reader reader = ReadersAndWriters.getUtf8Reader(f);){
            props.load(reader);
        }
        return props;
    }

    public static void writePropertiesToUtf8File(Properties properties, File f) throws IOException {
        Files.createDirectories(f.toPath().getParent(), new FileAttribute[0]);
        try (Writer writer = ReadersAndWriters.getUtf8Writer(f);){
            properties.store(writer, "");
        }
    }

    public static long spaceUsedInDir(File dir) {
        long sum = 0L;
        for (File f : dir.listFiles()) {
            if (f.isDirectory()) {
                sum += FileTools.spaceUsedInDir(f);
                continue;
            }
            sum += f.length();
        }
        return sum;
    }

    public static long getFileCreationDate(File f) throws IOException {
        try {
            Path file = f.toPath();
            BasicFileAttributes attr = Files.readAttributes(file, BasicFileAttributes.class, new LinkOption[0]);
            return attr.creationTime().toMillis();
        }
        catch (InvalidPathException e) {
            log.warn("Could no read file creation date of " + f.getAbsolutePath(), (Throwable)e);
            return 0L;
        }
    }

    public static FileAndAge creationDateMin(File fileOrDir, FileFilter fileFilter) throws IOException {
        if (fileOrDir.isFile()) {
            return new FileAndAge(fileOrDir, FileTools.getFileCreationDate(fileOrDir));
        }
        FileAndAge max = new FileAndAge(null, 0L);
        File[] subFileOrDirs = fileOrDir.listFiles(fileFilter);
        if (subFileOrDirs != null) {
            for (File f : subFileOrDirs) {
                FileAndAge sub = FileTools.creationDateMin(f, fileFilter);
                if (sub.age <= max.age) continue;
                max = sub;
            }
        }
        return max;
    }

    public static boolean isYoungerThanOrSameAgeAs(File derived, File source) {
        return source.lastModified() <= derived.lastModified();
    }

    public static boolean containsMatchingFile(File dir, FileFilter fileFilter) {
        assert (dir.exists());
        assert (dir.isDirectory());
        assert (fileFilter != null);
        File[] files = dir.listFiles();
        if (files != null) {
            for (File f : files) {
                if (!fileFilter.accept(f)) continue;
                return true;
            }
        }
        return false;
    }

    static {
        EXTENSIONS_TEXT = Sets.newHashSet((Object[])new String[]{"txt", "pom", "properties", "xml", "classpath", "project"});
        FILEFILTER_ALL_DIRECTORIES = new FileFilter(){

            @Override
            public boolean accept(File f) {
                return f.isDirectory();
            }
        };
        GENERATED_FILES_ENDINGS = new String[]{".DS_Store", "Hashes.db"};
        GENERATED_FILES_PARTS = new String[]{"/target"};
        IGNORE_GENERATED_FILES = new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                String path = pathname.getAbsolutePath();
                for (String end : GENERATED_FILES_ENDINGS) {
                    if (!path.endsWith(end)) continue;
                    return false;
                }
                return true;
            }
        };
        IGNORE_GENERATED_FILES_AND_DIRS = new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                String path = pathname.getAbsolutePath();
                for (String end : GENERATED_FILES_ENDINGS) {
                    if (!path.endsWith(end)) continue;
                    return false;
                }
                for (String part : GENERATED_FILES_PARTS) {
                    if (!path.contains(part)) continue;
                    return false;
                }
                return true;
            }
        };
        log = LoggerFactory.getLogger(FileTools.class);
        NO_FILES_ALL_DIRS_FILTER = new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                return pathname.isDirectory();
            }
        };
    }

    public static class WrappedInputStream
    extends InputStream {
        private final Function<Void, Void> closeFn;
        private final InputStream stream;

        public WrappedInputStream(InputStream stream, Function<Void, Void> closeFn) {
            this.closeFn = closeFn;
            this.stream = stream;
        }

        @Override
        public void close() throws IOException {
            this.stream.close();
            this.closeFn.apply(null);
        }

        @Override
        public int read() throws IOException {
            return this.stream.read();
        }

        @Override
        public int read(byte[] b) throws IOException {
            return this.stream.read(b);
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            return this.stream.read(b, off, len);
        }

        @Override
        public long skip(long n) throws IOException {
            return this.stream.skip(n);
        }

        @Override
        public int available() throws IOException {
            return this.stream.available();
        }

        @Override
        public synchronized void mark(int readlimit) {
            this.stream.mark(readlimit);
        }

        @Override
        public synchronized void reset() throws IOException {
            this.stream.reset();
        }

        @Override
        public boolean markSupported() {
            return this.stream.markSupported();
        }
    }

    public static interface IFileProcessor {
        public void process(File var1) throws IOException;
    }

    public static interface IDirProcessor {
        public void process(File var1) throws IOException;
    }

    public static class FileAndAge {
        long age;
        File f;

        public FileAndAge(File f, long lastModified) {
            this.f = f;
            this.age = lastModified;
        }

        public long getAge() {
            return this.age;
        }

        public File getFile() {
            return this.f;
        }

        public String toString() {
            return new Date(this.age) + " (" + this.age + ") = " + (this.f == null ? "--" : this.f.getAbsolutePath());
        }
    }
}

